function y=betaiinv(p,q,x)
% BETAIINV provides the inverse of the incomplete beta ratio function.
%
%          Y = BETAIINV(P,Q,X)
%
%          Given values (0)X(1), P(>0) and Q(>0), BETAIINV computes
%          the value of Y satisfying the relation
%                        
%                         1      y   p-1      q-1
%          I (p,q) =  ---------     t    (1-t)    dt = x
%           y         Beta(p,q)  0 
%
%          Thus, if X is the value of the lower tail area of the beta
%          distribution with P and Q degrees of freedom, Y is the quantile.
%
%          Default accuracy is about six decimal places (see TOL).

%          BETAIINV invokes BETA for a Newton-Raphson iteration.
%          The inverse incomplete beta ratio function is evaluated by using
%          the algorithm described in:
%          P.Griffiths, I.D.Hill (eds),'Applied Statistics Algorithms',
%          Royal Statistical Society 1985, Algorithm AS 64, 121-125
%

%          R.Marbach,  Ver.1.0,  Mar.07,1990

if (nargin~=3)
  error('Wrong number of input arguments')
else

 if any(size(p)~=size(q)) | ...
    any(size(x)~=size(p)) | ...
    any(size(x)~=size(q))
   error('input arguments don''t match')
 end

 if any(any(p <= 0)) | any(any(q <= 0))
  error('input parameter(s) out of range: p > 0, q > 0')
 end

 if any(any(x < 0)) | any(any(x > 1))
  error('input argument(s) out of range: 0  x  1')
 end

 % save input format of x in variable y
 [m,n]=size(x);
 y = zeros(m,n);
 % strung out as column vectors
 xc = x(:);
 pc = p(:);
 qc = q(:);
 [m,n]=size(xc);
 yc = zeros(m,n);
 Zero=zeros(m,n);
 One =ones(m,n);

 % betai=0
 index0 = (xc == 0);
 if any(index0), yc(index0) = Zero(index0); end
 % betai=1
 index1 = (xc == 1);
 if any(index1), yc(index1) = One(index1); end

 indexwork = ~(index0 | index1);
 if any(indexwork)
   xtmp=xc(indexwork);
   ptmp=pc(indexwork);
   qtmp=qc(indexwork);
   ytmp=yc(indexwork);

   % set accuracy to desired level
   % if d, say, is the number of accurate decimal places
   % desired, the exponent of TOL should be set at -2d-2.
   tol=1E-14;

   % set maximal number of iterations
   it1max=20; % newton steps
   it2max=150; % inner loops

   % set constants
   lower=0.0001;
   upper=0.9999;
   c1 = 2.30753;
   c2 = 0.27061;
   c3 = 0.99229;
   c4 = 0.04481;
   lnbeta=log(beta(ptmp,qtmp));

   %
   % allocate temporary variables
   %
   [m,n]=size(xtmp);
   pp = zeros(m,n);
   qq = zeros(m,n);
   a = zeros(m,n);
   r = zeros(m,n);
   s = zeros(m,n);
   t = zeros(m,n);
   h = zeros(m,n);
   w = zeros(m,n);
   z = zeros(m,n);
   zprev = zeros(m,n);
   prev = zeros(m,n);
   sq = zeros(m,n);
   g = zeros(m,n);
   tx = zeros(m,n);
   adj = zeros(m,n);

   %
   % change tail if necessary
   %
   indextail= (xtmp <= 0.5);
   if any(indextail)
     a(indextail) =xtmp(indextail);
     pp(indextail)=ptmp(indextail);
     qq(indextail)=qtmp(indextail);
     %INDEX=false
   end
   if any(~indextail)
     a(~indextail) =1-xtmp(~indextail);
     pp(~indextail)=qtmp(~indextail);
     qq(~indextail)=ptmp(~indextail);
     %INDEX=true
   end

   %
   % calculate the initial approximation of ytmp
   %
   r=sqrt(-log(a.*a));
   z=r-(c1+c2.*r)./(1+(c3+c4.*r).*r);
   if1index=((pp > 1) & (qq > 1));
   if any(if1index)
     r(if1index)=(z(if1index).*z(if1index) -3)./6;
     s(if1index)=(pp(if1index)+pp(if1index)-1).\1;
     t(if1index)=(qq(if1index)+qq(if1index)-1).\1;
     h(if1index)=(s(if1index)+t(if1index)).\2;
     w(if1index)=z(if1index).*sqrt( h(if1index)+r(if1index) )./h(if1index) ...
                 - (t(if1index)-s(if1index)).*( r(if1index)+(5/6) ...
                 - (3.*h(if1index)).\2 );
     ytmp(if1index)=pp(if1index) ./ ...
                      ( pp(if1index)+qq(if1index).*exp(2*w(if1index)) );
   end
   if any(~if1index)
     r(~if1index)=qq(~if1index)+qq(~if1index);
     t(~if1index)=(9.*qq(~if1index)).\1;
     t(~if1index)= r(~if1index) ...
                  .*(1-t(~if1index)+z(~if1index).*sqrt(t(~if1index))) ...
                  .*(1-t(~if1index)+z(~if1index).*sqrt(t(~if1index))) ...
                  .*(1-t(~if1index)+z(~if1index).*sqrt(t(~if1index)));
     if2index=~if1index & (t <= 0);
     if any(if2index)
       ytmp(if2index)=1-exp((log((1-a(if2index)).*qq(if2index)) ...
                      +lnbeta(if2index))./qq(if2index));
     end
     if any(~if2index)
       t(~if2index)=(4.*pp(~if2index)+r(~if2index)-2)./t(~if2index);
       if3index=~if2index & (t <= 1);
       if any(if3index)
         ytmp(if3index)=exp((log(a(if3index).*pp(if3index)) ...
                        + lnbeta(if3index))./pp(if3index));
       end
       if any(~if3index)
         ytmp(~if3index)=1-(t(~if3index)+1).\2;
       end
     end %~if2index
   end %~if1index

   %
   % Solving for ytmp by a modified Newton-Raphson method,
   % using the M-file BETA
   %
   r=1-pp;
   t=1-qq;
   zprev=zeros(m,n);
   sq   =ones(m,n);
   prev =ones(m,n);
   if1index= (ytmp < lower);
   if any(if1index), ytmp(if1index)=lower; end
   if1index= (ytmp > upper);
   if any(if1index), ytmp(if1index)=upper; end

   % Newton-Raphson iteration
   for i=1:it1max
     % calling BETA
     z=beta(pp,qq,ytmp);
     z=(z-a).*exp( lnbeta + r .*log(ytmp) + t.*log(1-ytmp) );
     if1index=( (z.*zprev) <= 0);
     if any(if1index), prev(if1index)=sq(if1index); end
     g=ones(m,n);   

     % inner iteration loop
     for ii=1:it2max
       % initialize 
       runindex=ones(m,n);
       tx= - ones(m,n);
       adj(runindex)=g(runindex).*z(runindex);
       sq(runindex)=adj(runindex).*adj(runindex);
       if1index = runindex & (sq >= prev);
       if any(if1index)
         g(if1index)=g(if1index)./3;
       end;
       if1indexN= runindex & ~(sq >= prev);
       if any(if1indexN)
         tx(if1indexN) = ytmp(if1indexN) - adj(if1indexN);
       end
       % update runindex
       runindex = runindex & ~( (tx >= 0) & (tx <= 1) );
       % watch for break
       if ~any(runindex), break, end
       if any(runindex & if1indexN)
         g(runindex & if1indexN)=g(runindex & if1indexN)./3;
       end
       if (ii==it2max)
         disp('Warning BETAIINV: number of inner iterations IT2MAX exceeded')
       end
     end %ii

     if all(prev <= tol), break, end
     if all( (z.*z) <= tol), break, end
     if1index= (tx == 0) | (tx == 1);
     if any(if1index)
       g(if1index)=g(if1index)./3;
       % repeat inner iteration loop
       for ii=1:it2max
         % initialize
         runindex=if1index;
         tx= - ones(m,n);
         adj(runindex)=g(runindex).*z(runindex);
         sq(runindex)=adj(runindex).*adj(runindex);
         if2index = runindex & (sq >= prev);
         if any(if2index)
           g(if2index)=g(if2index)./3;
         end;
         if2indexN= runindex & ~(sq >= prev);
         if any(if2indexN)
           tx(if2indexN) = ytmp(if2indexN) - adj(if2indexN);
         end
         % update runindex
         runindex = runindex & ~( (tx >= 0) & (tx <= 1) );
         % watch for break
         if ~any(runindex), break, end
         if any(runindex & if2indexN)
           g(runindex & if2indexN)=g(runindex & if2indexN)./3;
         end
         if (ii==it2max)
           disp('Warning BETAIINV: number of inner iterations IT2MAX exceeded')
         end
       end %ii
     end %if1index
     if all(prev <= tol), break, end
     if all( (z.*z) <= tol), break, end

     if all(tx == ytmp), break, end
     ytmp=tx;
     zprev=z;
     if (i==it1max)
       disp('Warning BETAIINV: number of Newton iterations IT1MAX exceeded')
     end
   end %i

   % re-change tails if necessary
   if any(~indextail)
     ytmp(~indextail)=1-ytmp(~indextail);
   end
 end %indexwork
 
 % restore results into required shape
 yc(indexwork)=ytmp;
 y(:) = yc;
end
