function [u,b,w] = lanbi(x,y,useR,flag)
%LANBI  Lanczos bidiagonalization.
%
%             [U,B,W] = LANBI(X,Y)
%
%       provides the Lanczos bidiagonalization X = U B W' using
%       the unit-length scaled vector X'*Y as start vector w1.
%       U and W are orthonormal, B is an upper bidiagonal matrix.
%       Only the PLS1 algorithm is implemented, so Y has to be a vector!
%
%       Several stopping criteria are used. Either ..
%         (1) The maximum rank = min(size(X)) is attained or
%         (2) The norm of either U(:,r) or W(:,r) falls short of TOL1 or
%         (3) Y' (I-UU') Y  <  TOL2*Y'Y   (i.e., SSE < TOL2*TSSadj).
%
%             [U,B,W] = LANBI(X,Y,USER)
%
%       stops bidiagonalization at rank USER (if the stopping criteria permit).
%
%       By using LANBI(X,Y,'noprint') and LANBI(X,Y,USER,'noprint'), the 
%       display of the status line showing the actual stopping criterium
%       used, is suppressed.

%     R. Marbach,  Ver.1.0,  Mar.24,1990

if nargout~=3
  error('Wrong number of output arguments')
end
if nargin==2
  noprint=0;
  useR=-1;
elseif nargin==3 & isstr(useR)
  noprint=1;
  useR=-1;
elseif nargin==3
  noprint=0;
elseif nargin==4 & isstr(flag)
  noprint=1;
else
  error('Too much or too few input arguments')
end
if min(size(y))~=1
  error('Only PLS1 is implemented => Y has to be a vector')
end
y=y(:);

[m,n]=size(x);
% determination of maximal rank (considering mean-centered data)
% and pre-allocating variables
if norm(mean(x)) < sqrt(n)*eps
  maxR=min(m-1,n);
else
  maxR=min(m,n);
end
% user selected rank
if useR > 0, maxR=min(maxR,useR); end
lw = zeros(maxR,1);
lu = zeros(maxR,1);
w  = zeros(n,maxR);
u  = zeros(m,maxR);

% set default values of tolerances
tol1=sqrt(n)*2.4E-7;    % Perkin-Elmer: tol1=sqrt(n)*1E-4;
tol2=1E-5;              % SSE (consistent solution)


TSS = y'*y;
SSE = TSS;
% ------------ PLS iteration ------------
R=1;
while (1==1) %dummy
  if R==1
   % PLS-start vector(s)
   w(:,R)=x'*y;
     % 12 July 1993
     % insect start *******
     %
     %  w(1:72,R)=zeros(72,1);
     %
     % insect end   *******
   lw(R)=norm(w(:,R)); w(:,R)=w(:,R)/lw(R);
   u(:,R)=x*w(:,R);
   lu(R)=norm(u(:,R)); u(:,R)=u(:,R)/lu(R);
  else
   % new w
   w(:,R) = x'*u(:,R-1) - lu(R-1)*w(:,R-1);
   for r=1:R-1, w(:,R)=w(:,R) - (w(:,R)'*w(:,r))*w(:,r); end
   lw(R)=norm(w(:,R)); w(:,R) = w(:,R)/lw(R);
   % new u
   u(:,R) = x*w(:,R) - lw(R)*u(:,R-1);
   for r=1:R-1, u(:,R)=u(:,R) - (u(:,R)'*u(:,r))*u(:,r); end
   lu(R)=norm(u(:,R)); u(:,R) = u(:,R)/lu(R);
  end 
  % loop control
  if (lw(R) < tol1) | (lu(R) < tol1)
   if ~noprint
    fprintf(['bidiagonalization stop: Minute lu or lw at R = %2.0f\n',7,7,7],R)
   end
   break
  end;
  SSE = SSE - (y'*u(:,R))*(y'*u(:,R));
  if SSE < tol2*TSS
   if ~noprint
    fprintf(['bidiagonalization stop: Consistent at R = %2.0f\n',7,7,7],R)
   end
   break
  end
  if R >= maxR
   if ~noprint
    fprintf('bidiagonalization stopped regularly at R = %2.0f\n',R)
   end
   break
  end
  R=R+1;
end

% generate 'economy version' of PLS decomposition
b = diag(lu(1:R));
if R > 1
  b = b + diag(lw(2:R),1);
end
u = u(:,[1:R]);
w = w(:,[1:R]);
