function rms=plsval(x,y,xval,yval,useR)
%PLSVAL rms-prediction error [mg/dL] of validation set VS. PLS-rank.
%
%          RMS = PLSVAL(X,Y,XVAL,YVAL)         or
%          RMS = PLSVAL(X,Y,XVAL,YVAL,USER)
%
%       The PLS-inverses of mean-centered regression matrix X and vector Y
%       with inversion ranks R=0..min{USER; max. rank determined by LANBI},
%       are validated against validation set XVAL, YVAL.
%       Only PLS1 is implemented, so Y and YVAL have to be vectors.
%
%       In order to detect pre-mature PLS stopping: size(RMS)=[2,useR+1] with:
%
%               RMS(0)    RMS(1)  ..  RMS(Rpls)  RMS(Rpls) .. RMS(Rpls)
%       RMS =                                                          
%             row(XVAL) row(XVAL) ..  row(XVAL)     0      ..    0     
%
%       the first column meaning 'prediction by average (calibration) value.'


% R. Marbach,  11-Jan-1994; numerically tested using PLSPRED 10-Jan-94

if nargout~=1
  error('Wrong number of output arguments')
end
if (nargin~=4) & (nargin~=5)
  error('Wrong number of input arguments')
end
if min(size(y)) >1 | min(size(yval)) > 1
  error('Y and YVAL have to be vectors.')
end
if row(x)~=length(y) | row(xval)~=length(yval)
  error('Check dimensions of X,Y and/or XVAL,YVAL.')
end
if col(x)~=col(xval)
  error('Calibration and validation spectra are diff. in wavelength.')
end

% maximum rank
Rmax=min( row(x)-1, col(x) );
if nargin==4, useR=Rmax; end
if useR>Rmax, useR=Rmax; end
if (max(size(useR)) > 1) | (useR < 1)
  error('Check input argument USER.')
end

% some all-data statistics
M=row(x);       Mval=row(xval);
y=y(:);         yval=yval(:);
yav = mean(y);
xav = mean(x);

% PLS-decomposition
[u,b,w] = lanbi(x-ones(M,1)*xav, y-yav, useR);
Rpls=row(b);    % max. PLS-rank determined by LANBI

% allocating
prd=zeros(Mval,Rpls);
B=zeros(col(x),Rpls);
for r=1:Rpls
 B(:,r)=w(:,1:r)*inv(b(1:r,1:r))*(u(:,1:r)'*(y-yav));
 prd(:,r)=yav + (xval-ones(Mval,1)*xav)*B(:,r);
end
% append PLS-inversion rank=0
prd=[ones(Mval,1)*yav, prd];
err=yval*ones(1,Rpls+1) - prd;   % using 'err' earlier invokes compiler errors!
% optional:
% save c:\matlab\user\val_err err
% fprintf('matrix ''err'' of prediction errors (Mval-by-R) saved\n')
% fprintf('in MAT:USER:VAL_ERR.MAT; remember Ypred = Ytrue - err !\n')
if Mval > 1
 rms=sqrt(sum(err.^2)/Mval);  % "sum" would work row-wise for vectors
else
 rms=abs(err);
end
% outputting
rms=[rms; ones(size(rms))*Mval];
if Rpls < Rmax
  for m=1:Rmax-Rpls
    rms=[rms, [rms(1,Rpls+1); 0] ];
  end
end